/**
  @file             SM_HLL.c
  @version          0.1.0.0          

  @brief            
  @details          

  Project           Stepper Motor Driver
  Platform          MPC5606S

  SWVersion         0.1.0


  (c) Copyright 2011 Freescale Semiconductor Inc.
  All Rights Reserved.
*/
/*==================================================================================================
==================================================================================================*/

#ifdef __cplusplus
extern "C"{
#endif

/*==================================================================================================
                                         INCLUDE FILES
==================================================================================================*/

#include "MAC57D54H.h"
#include "SM_Tables.h"
#include "SM_Global.h"			/* Includes Stepper motor driver general definitions */
#include "SM_HLL.h"				/* Includes Higl Level Layer */ 
#include "SM_LLL.h"				/* Includes Low Level Layer */
#include "SM_RAL.h"				/* Includes Register Access Layer */

/*==================================================================================================
                                       GLOBAL VARIABLES
==================================================================================================*/

SM_MotorParams_t SM_HLL_rMtr[SM_MAX_MOTORS];	/* Array of structures containing driver parameters */

/*==================================================================================================
                                       GLOBAL CONSTANTS
==================================================================================================*/

uint16_t afr14Ramp[256]=
{
	3393,6286,6906,7217,7406,7533,7625,7694,7748,7792,7827,7857,7883,7904,7923,7940,7954,7967,7979,7990,
	7999,8008,8016,8023,8030,8036,8042,8047,8052,8057,8061,8065,8069,8072,8076,8079,8082,8085,8088,8090,
	8093,8095,8097,8099,8101,8103,8105,8107,8109,8110,8112,8114,8115,8116,8118,8119,8120,8122,8123,8124,
	8125,8126,8127,8128,8129,8130,8131,8132,8133,8134,8135,8135,8136,8137,8138,8138,8139,8140,8140,8141,
	8142,8142,8143,8143,8144,8145,8145,8146,8146,8147,8147,8148,8148,8149,8149,8149,8150,8150,8151,8151,
	8152,8152,8152,8153,8153,8153,8154,8154,8155,8155,8155,8156,8156,8156,8156,8157,8157,8157,8158,8158,
	8158,8158,8159,8159,8159,8160,8160,8160,8160,8161,8161,8161,8161,8161,8162,8162,8162,8162,8163,8163,
	8163,8163,8163,8164,8164,8164,8164,8164,8165,8165,8165,8165,8165,8165,8166,8166,8166,8166,8166,8166,
	8167,8167,8167,8167,8167,8167,8168,8168,8168,8168,8168,8168,8168,8168,8169,8169,8169,8169,8169,8169,
	8169,8170,8170,8170,8170,8170,8170,8170,8170,8170,8171,8171,8171,8171,8171,8171,8171,8171,8171,8172,
	8172,8172,8172,8172,8172,8172,8172,8172,8172,8173,8173,8173,8173,8173,8173,8173,8173,8173,8173,8173,
	8173,8174,8174,8174,8174,8174,8174,8174,8174,8174,8174,8174,8174,8175,8175,8175,8175,8175,8175,8175,
	8175,8175,8175,8175,8175,8175,8175,8176,8176,8176,8176,8176,8176,8176,8176,8176
};	/* Ramp - up/down table of 14bit fractionals */

/*==================================================================================================
                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/

static void SM_HLL_TimerInit(SM_MotorId_t eMtrId);		/* The function initializes internal timers */
static void SM_HLL_TimerDisable(SM_MotorId_t eMtrId);	/* The function disables internal timers */
static void SM_HLL_InitHwModules(SM_MotorId_t eMtrId);	/* The function initializes internal HW modules */
static void SM_HLL_SsdInit(SM_MotorId_t eMtrId);		/* The function initializes SSD module for RTZ */	
static uint32_t SM_HLL_MoveAlgorithm(SM_MotorParams_t *prMtr);	/* The function executes motor algorithm */
static void SM_HLL_RtzMove(SM_MotorParams_t *prMtr);			/* The function executes RTZ movement */
static uint32_t SM_HLL_MoveScheduler(SM_MotorParams_t *prMtr);	/* The function schedules SM operations */

/*==================================================================================================
                                       LOCAL FUNCTIONS
==================================================================================================*/

/*******************************************************************************
* @function_name SM_Sqrt
*//*
* @brief 		The function calculates square root of the input value	
* @param[in] 	u32Inp	- Input value 
* @retval 		u32Xn 	- Calculated square root
* @details 	  	The function calculates square root of integer number. Function uses 
*				Newton's method but doesn't use float types so the accuracy is 
*				limited. Function should be replaced. 
*/
static uint32_t SM_Sqrt(uint32_t u32Inp)
{
	uint32_t u32Xn,u32XnM1; 	
	
	u32Xn = 32768; 			/* initial value */ 
	
	do
	{
		u32XnM1 = u32Xn;	/* save previous iteration */ 
		/* calculate actual iteration */ 	
		u32Xn = u32Xn - (((u32Xn * u32Xn) - u32Inp) / (2 * u32Xn));
	
	/* iterates until actual an previous iteration match */	
	}while(u32XnM1 != u32Xn);	
		
	return u32Xn;	/* return last iteration value */		
}

/*******************************************************************************
* @function_name SM_HLL_TimerInit
*//*
* @brief 		The function initializes timer which is used by Stepper Motor 
*				algorithm for periodic update.
* @param[in] 	eMtrId 	- ID of the motor going to be affected
* @details 		The function calls function SM_LLL_TimerInit in Low SW layer
* 				to initialize timer of the corresponding motor.  
*/
static void SM_HLL_TimerInit(SM_MotorId_t eMtrId)
{
	SM_LLL_TimerInit(eMtrId);		
}

/*******************************************************************************
* @function_name SM_HLL_TimerDisable
*//*
* @brief 		The function disables timer which is used by Stepper Motor
*				algorithm for periodic update.	
* @param[in]  	eMtrId 	- ID of the motor going to be affected
* @details 		The function calls function SM_LLL_TimerDisable in Low SW layer
*				to disable timer of the corresponding motor.
*/
static void SM_HLL_TimerDisable(SM_MotorId_t eMtrId)
{
	SM_LLL_TimerDisable(eMtrId);		
}

/*******************************************************************************
* @function_name SM_HLL_InitHwModules
*//*
* @brief 		The function initializes HW modules of the corresponding motor 		
* @param[in] 	eMtrId 	- ID of the motor going to be initialized 
* @details 		The function calls function SM_LLL_InitHwModules in Low SW layer
*				to initialize HW modules (SMC, SIUL & INTC) of the corresponding
*				motor
*/
static void SM_HLL_InitHwModules(SM_MotorId_t eMtrId)
{
	SM_LLL_InitHwModules(&SM_HLL_rMtr[eMtrId]);		
}

/*******************************************************************************
* @function_name SM_HLL_SsdInit
*//*
* @brief 		The function initializes SSD module for the RTZ movement
* @param[in] 	eMtrId 	- ID of the motor going to be initialized for RTZ movement 
* @details 		The function calls function SM_LLL_SsdInit that initializes SSD & INTC
*				modules for RTZ movement of the motor defined by eMtrId. 
*/
static void SM_HLL_SsdInit(SM_MotorId_t eMtrId)
{
	SM_LLL_SsdInit(&SM_HLL_rMtr[eMtrId]);		
}

/*******************************************************************************
* @function_name StepPosUpdate
*//*
* @brief 		The function increments/decrements actual motor position according to 
*				the required movement direction.
* @param[in] 	prMtr - pointer to structure containing all the motor parameters
* @details 		The function increments stepper motors position whent the required 
*				movement direction is clockwise. Otherwise it decrements the actual 
*				motor position.  
*/

static void StepPosUpdate(SM_MotorParams_t *prMtr)
{
	/* If is the Command Direction clockwise */
	if(prMtr->trMtrAlgo.eMtrCmdDir == SM_DIRECTION_CW) 
	{
		/* Increment actual position */
		prMtr->trMtrAlgo.s32Pos = prMtr->trMtrAlgo.s32Pos + 1;
	}
	
	/* If is tne Command Direction counterclockwise */
	else 
	{
		/* Decrement actual position */
		prMtr->trMtrAlgo.s32Pos = prMtr->trMtrAlgo.s32Pos - 1;
	}
				
	/* If the Command Direction equals to the motor Stop Direction */			
	if(prMtr->trMtrAlgo.eMtrCmdDir != prMtr->trGeneral.eMotorNominalDir)
	{		
	   	/* Decrement u16Step */
		prMtr->trMtrAlgo.u16Step--;
		
		/*If the stepper motor exceeded Sin/Cos table length (>0) */
    	if(prMtr->trMtrAlgo.u16Step >= HLL_TABLELENGTH)
    	{
    		/* Add table length to the u16Step */
    		prMtr->trMtrAlgo.u16Step += HLL_TABLELENGTH;
    	}
	}
    
    /* If the Command Direction doesnt equal to the motor Stop Direction */
    else
    {
		/* Increment u16Step */
		prMtr->trMtrAlgo.u16Step++;
		
		/* If the u16Step exceeded Sin/Cos table length */
		if(prMtr->trMtrAlgo.u16Step >= HLL_TABLELENGTH)
		{
			/* Substract table length from the u16Step */
			prMtr->trMtrAlgo.u16Step -= HLL_TABLELENGTH;		
		}
    }
}

/*******************************************************************************
* @function_name SM_HLL_MoveAlgorithm
*//*
* @brief 		The function operates motor movement algorithm 	
* @param[in] 	prMtr - pointer to structure containing all the motor parameters
* @retval 		u32NextStepTimeout	- next ustep timeout
* @details 		The function updates stepper motor position, actual microstep and 
*				next microstep timeout.
*/
static uint32_t SM_HLL_MoveAlgorithm(SM_MotorParams_t *prMtr)
{    
 	uint32_t u32Temp;
 	int32_t s32TmpStp=0,s32TmpDiff=0; 
			
	if(prMtr->trGeneral.eAlgoType == SM_ALGO_VAR_STEP_TIME)
	{	
		if(prMtr->trGeneral.eCmdToAlgo == SM_CMD_TO_ALGO_UPDATE)
		{
			prMtr->trGeneral.eCmdToAlgo = SM_CMD_TO_ALGO_NONE;
			
			if((prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_STOP) | \
				(prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_FROZEN))
			{
				prMtr->trMtrAlgo.u16Ramp = 0u;
				prMtr->trMtrAlgo.s32CmdPos = prMtr->trGeneral.s32ReqPos;
				
				prMtr->trMtrAlgo.u32MaxVelTimeout = prMtr->trGeneral.u32ReqMaxVelTimeout;
				
				/* Motor starts movement at the minimum speed */ 
				prMtr->trMtrAlgo.u32ActVelTimeout = prMtr->trGeneral.u32StartVelTimeout;
				
				if(prMtr->trMtrAlgo.s32Pos < prMtr->trMtrAlgo.s32CmdPos)
				{
					prMtr->trMtrAlgo.eMtrCmdDir = SM_DIRECTION_CW;
					prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPUP;
				}
				
				else if(prMtr->trMtrAlgo.s32Pos > prMtr->trMtrAlgo.s32CmdPos)
				{
					prMtr->trMtrAlgo.eMtrCmdDir = SM_DIRECTION_CCW;
					prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPUP;
				}
				
				else 
				{
					prMtr->trGeneral.eMotorState = SM_MOTOR_STATE_STOP;
				}
			}
			
			else if(prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_RUN)
			{
				if(prMtr->trGeneral.s32ReqPos > (prMtr->trMtrAlgo.s32Pos + prMtr->trMtrAlgo.u16Ramp))
				{
					if(prMtr->trMtrAlgo.u32MaxVelTimeout < prMtr->trGeneral.u32ReqMaxVelTimeout)
					{
						prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_SLOWDOWN;
					}
					
					else if(prMtr->trMtrAlgo.u32MaxVelTimeout > prMtr->trGeneral.u32ReqMaxVelTimeout)
					{
						prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPUP;
					}
					
					else
					{
						/* do nothing */	
					}
				}
				
				else
				{
					prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPDOWN;	
				}
				
				prMtr->trMtrAlgo.s32CmdPos = prMtr->trGeneral.s32ReqPos;
				prMtr->trMtrAlgo.u32MaxVelTimeout = prMtr->trGeneral.u32ReqMaxVelTimeout;
			}
			
			else if((prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_RAMPUP) | \
				(prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_RAMPDOWN) | \
				(prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_SLOWDOWN))
			{
				if(prMtr->trMtrAlgo.eMtrCmdDir == SM_DIRECTION_CW)
				{
					if(prMtr->trGeneral.s32ReqPos > (prMtr->trMtrAlgo.s32Pos + prMtr->trMtrAlgo.u16Ramp))
					{
						if(prMtr->trMtrAlgo.u32ActVelTimeout < prMtr->trGeneral.u32ReqMaxVelTimeout)
						{
							prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_SLOWDOWN;	
						}
						
						else if(prMtr->trMtrAlgo.u32ActVelTimeout < prMtr->trGeneral.u32ReqMaxVelTimeout)
						{
							prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPUP;
						}
						
						else
						{
							prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPUP;	
						}
					}
					
					else
					{	
						prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPDOWN;
					}		
				}
				
				else
				{
					if(prMtr->trGeneral.s32ReqPos < (prMtr->trMtrAlgo.s32Pos - prMtr->trMtrAlgo.u16Ramp))
					{
						if(prMtr->trMtrAlgo.u32ActVelTimeout < prMtr->trGeneral.u32ReqMaxVelTimeout)
						{
							prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_SLOWDOWN;	
						}
						
						else if(prMtr->trMtrAlgo.u32ActVelTimeout < prMtr->trGeneral.u32ReqMaxVelTimeout)
						{
							prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPUP;	
						}
						
						else
						{
							prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPUP;	
						}
					}
					
					else
					{	
						prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPDOWN;
					}				
				}
							
				prMtr->trMtrAlgo.s32CmdPos = prMtr->trGeneral.s32ReqPos;
				prMtr->trMtrAlgo.u32MaxVelTimeout = prMtr->trGeneral.u32ReqMaxVelTimeout;
			}
			
			else if((prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_RAMPDOWN_DONE) | \
				(prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_RAMPUP_DONE))
			{
				prMtr->trMtrAlgo.s32CmdPos = prMtr->trGeneral.s32ReqPos;
				prMtr->trMtrAlgo.u32MaxVelTimeout = prMtr->trGeneral.u32ReqMaxVelTimeout;
			}
		}

		/* Algorithm !! */
	    /* Absolute position to the destination */
		if(prMtr->trMtrAlgo.s32Pos < prMtr->trMtrAlgo.s32CmdPos)
		{
			prMtr->trMtrAlgo.u32AbsPosToDest = prMtr->trMtrAlgo.s32CmdPos - prMtr->trMtrAlgo.s32Pos;
		}
			
		else if(prMtr->trMtrAlgo.s32Pos > prMtr->trMtrAlgo.s32CmdPos)
		{
			prMtr->trMtrAlgo.u32AbsPosToDest = prMtr->trMtrAlgo.s32Pos - prMtr->trMtrAlgo.s32CmdPos;
		} 
			
		else
		{
			prMtr->trMtrAlgo.u32AbsPosToDest = 0;	
		}
		        
		    
		/* Algorithm state machine */
	   	if(prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_STOP)
		{
			prMtr->trGeneral.eMotorState = SM_MOTOR_STATE_STOP;
		}
		    
		else if(prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_RAMPUP)
		{
			if(prMtr->trMtrAlgo.u32ActVelTimeout > prMtr->trMtrAlgo.u32MaxVelTimeout)
			{			
				u32Temp = prMtr->trMtrAlgo.u32ActVelTimeout;
				
				if(prMtr->trMtrAlgo.u16Ramp > 255)
				{
					u32Temp *= afr14Ramp[255];
				}
				
				else
				{
					u32Temp *= afr14Ramp[prMtr->trMtrAlgo.u16Ramp];
				}
			
				u32Temp >>= 13;
				prMtr->trMtrAlgo.u32ActVelTimeout = u32Temp;
						
				if(prMtr->trMtrAlgo.u32ActVelTimeout <= prMtr->trMtrAlgo.u32MaxVelTimeout)
				{
					prMtr->trMtrAlgo.u32ActVelTimeout = prMtr->trMtrAlgo.u32MaxVelTimeout;	
					prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPUP_DONE;
				}
								
				else if(prMtr->trMtrAlgo.u32AbsPosToDest <= (prMtr->trMtrAlgo.u16Ramp + 2))
				{
					prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPDOWN;
				}
				
				else
				{
					prMtr->trMtrAlgo.u16Ramp++;	
				}
			}
		
			else
			{
				prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPUP_DONE;
			}
		
			StepPosUpdate(prMtr);
		}

		else if(prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_RAMPDOWN)
		{
			if(prMtr->trMtrAlgo.u32ActVelTimeout < prMtr->trGeneral.u32StartVelTimeout)
			{
				u32Temp = prMtr->trMtrAlgo.u32ActVelTimeout << 13;
				
				if(prMtr->trMtrAlgo.u16Ramp > 255)
				{
					u32Temp /= (afr14Ramp[255]);
				}
				
				else
				{
					u32Temp /= (afr14Ramp[prMtr->trMtrAlgo.u16Ramp]);
				}
				prMtr->trMtrAlgo.u32ActVelTimeout = u32Temp;
				
				if(prMtr->trMtrAlgo.u32ActVelTimeout >= prMtr->trGeneral.u32StartVelTimeout)
				{
					prMtr->trMtrAlgo.u32ActVelTimeout = prMtr->trGeneral.u32StartVelTimeout;
					prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPDOWN_DONE;	
				}
				
				else if(prMtr->trMtrAlgo.u16Ramp == 0)
				{
					prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPDOWN_DONE;
				}
				
				else
				{
					prMtr->trMtrAlgo.u16Ramp--;	
				}
			}
				
			else
			{
				prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPDOWN_DONE;	
			}
			
			StepPosUpdate(prMtr);
		}
		       
		else if(prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_RAMPUP_DONE)
		{
			prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RUN;
			StepPosUpdate(prMtr);
		}   
		    
		else if(prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_RAMPDOWN_DONE)
		{
		  	if(prMtr->trMtrAlgo.s32Pos < prMtr->trMtrAlgo.s32CmdPos)
			{
				prMtr->trMtrAlgo.eMtrCmdDir = SM_DIRECTION_CW;
				if(prMtr->trMtrAlgo.u32AbsPosToDest > 5)
				{
					prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPUP;
				}
				
				StepPosUpdate(prMtr);
				
				if(prMtr->trMtrAlgo.s32Pos == prMtr->trMtrAlgo.s32CmdPos)
				{
					prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_STOP;
					prMtr->trGeneral.eMotorState = SM_MOTOR_STATE_STOP;	
				}
			}
			
			else if(prMtr->trMtrAlgo.s32Pos > prMtr->trMtrAlgo.s32CmdPos)
			{
				prMtr->trMtrAlgo.eMtrCmdDir = SM_DIRECTION_CCW;
				if(prMtr->trMtrAlgo.u32AbsPosToDest > 5)
				{
					prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPUP;
				}
				
				StepPosUpdate(prMtr);
								
				if(prMtr->trMtrAlgo.s32Pos == prMtr->trMtrAlgo.s32CmdPos)
				{
					prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_STOP;
					prMtr->trGeneral.eMotorState = SM_MOTOR_STATE_STOP;	
				}
			}
					
			else 
			{
				prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_STOP;
				prMtr->trGeneral.eMotorState = SM_MOTOR_STATE_STOP;
			}			
		}
		     	    
		else if(prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_SLOWDOWN)
		{
		  	if(prMtr->trMtrAlgo.u32ActVelTimeout < prMtr->trMtrAlgo.u32MaxVelTimeout)
			{
				u32Temp = prMtr->trMtrAlgo.u32ActVelTimeout << 13;
				
				if(prMtr->trMtrAlgo.u16Ramp > 255)
				{
					u32Temp /= (afr14Ramp[255]);
				}
				
				else
				{
					u32Temp /= (afr14Ramp[prMtr->trMtrAlgo.u16Ramp]);
				}
				prMtr->trMtrAlgo.u32ActVelTimeout = u32Temp;
				
				if(prMtr->trMtrAlgo.u32ActVelTimeout >= prMtr->trMtrAlgo.u32MaxVelTimeout)
				{
					prMtr->trMtrAlgo.u32ActVelTimeout = prMtr->trMtrAlgo.u32MaxVelTimeout;
					prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RUN;	
				}
				
				else if(prMtr->trMtrAlgo.u16Ramp == 0)
				{
					prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPDOWN_DONE;
				}
				
				else
				{
					prMtr->trMtrAlgo.u16Ramp--;	
				}
			}
				
			else
			{
				prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RUN;	
			}
				
			StepPosUpdate(prMtr);
		}
		   
		else if(prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_RUN)
		{
			if(prMtr->trMtrAlgo.u32AbsPosToDest <= prMtr->trMtrAlgo.u16Ramp)
			{
				prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_RAMPDOWN;
			}
				
			StepPosUpdate(prMtr);
		}
			
		else
		{
			prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_STOP;
		}
		
		
		if(prMtr->trGeneral.eCmdToAlgo == SM_CMD_TO_ALGO_FREEZE)
		{
			if(prMtr->trMtrAlgo.eAlgoState == SM_ALGO_STATE_RAMPDOWN_DONE)
			{								
				prMtr->trGeneral.eFreeze = SM_FREEZE_STATE_FREEZE;
				
				prMtr->trMtrAlgo.eAlgoState = SM_ALGO_STATE_FROZEN;			
								
				prMtr->trGeneral.eCmdToAlgo = SM_CMD_TO_ALGO_NONE;
			}
		}
	}
	
	else if(prMtr->trGeneral.eAlgoType == SM_ALGO_VAR_STEP_SIZE)
	{
		prMtr->trMtrAlgo.s16TargetPos = prMtr->trGeneral.s32ReqPos;
		
		prMtr->trMtrAlgo.s16TempCalc1 += ((prMtr->trMtrAlgo.s16TargetPos - prMtr->trMtrAlgo.s16TempCalc1) >> prMtr->trMtrAlgo.u16DampingFactor);
		prMtr->trMtrAlgo.s16ActSpeed = (prMtr->trMtrAlgo.s16TempCalc1 - prMtr->trMtrAlgo.s16ActPos) >> prMtr->trMtrAlgo.u16DampingFactor;
		prMtr->trMtrAlgo.s16ActAcc = prMtr->trMtrAlgo.s16ActSpeed - prMtr->trMtrAlgo.s16PrevSpeed;
		
		if(prMtr->trMtrAlgo.s16PrevSpeed > 0)
		{
			if(prMtr->trMtrAlgo.s16ActAcc > prMtr->trMtrType.s16MotorMaxAcc)
			{
				prMtr->trMtrAlgo.s16ActSpeed = prMtr->trMtrAlgo.s16PrevSpeed + prMtr->trMtrType.s16MotorMaxAcc;
			}
			
			if(prMtr->trMtrAlgo.s16ActAcc < (-prMtr->trMtrType.s16MotorMaxDec))
			{
				prMtr->trMtrAlgo.s16ActSpeed = prMtr->trMtrAlgo.s16PrevSpeed - prMtr->trMtrType.s16MotorMaxDec;
			}
		}
		
		else 
		{
			if(prMtr->trMtrAlgo.s16ActAcc < (-prMtr->trMtrType.s16MotorMaxAcc))
			{
				prMtr->trMtrAlgo.s16ActSpeed = prMtr->trMtrAlgo.s16PrevSpeed - prMtr->trMtrType.s16MotorMaxAcc;
			}
			
			if(prMtr->trMtrAlgo.s16ActAcc > prMtr->trMtrType.s16MotorMaxDec)
			{
				prMtr->trMtrAlgo.s16ActSpeed = prMtr->trMtrAlgo.s16PrevSpeed + prMtr->trMtrType.s16MotorMaxDec;
			}
		}
		
		if(prMtr->trMtrAlgo.s16ActSpeed >= 0)
		{
			if(prMtr->trMtrAlgo.s16ActSpeed > prMtr->trMtrType.s16MotorMaxVel)
			{
				prMtr->trMtrAlgo.s16ActSpeed = prMtr->trMtrType.s16MotorMaxVel;
			}
		}
		
		else
		{
			if((-prMtr->trMtrAlgo.s16ActSpeed) > prMtr->trMtrType.s16MotorMaxVel)
			{
				prMtr->trMtrAlgo.s16ActSpeed = (-prMtr->trMtrType.s16MotorMaxVel);
			}
		}
		
		prMtr->trMtrAlgo.s16ActPos = prMtr->trMtrAlgo.s16OldPos + prMtr->trMtrAlgo.s16ActSpeed;
		prMtr->trMtrAlgo.s16PrevSpeed = prMtr->trMtrAlgo.s16ActSpeed;
		
		s32TmpDiff = prMtr->trMtrAlgo.s16ActPos - prMtr->trMtrAlgo.s16OldPos;
		
		s32TmpStp = s32TmpDiff/4;
		prMtr->trMtrAlgo.s16OldPos = prMtr->trMtrAlgo.s16OldPos + s32TmpStp;
		
		prMtr->trMtrAlgo.u16Step = prMtr->trMtrAlgo.s16OldPos % 128;
		
		
		if(prMtr->trMtrAlgo.s16ActSpeed > 0)
		{
			prMtr->trMtrAlgo.eMtrCmdDir = SM_DIRECTION_CW;
			
			if((prMtr->trMtrAlgo.s16ActSpeed <= prMtr->trMtrType.u16Threshold1) | (prMtr->trMtrAlgo.s16ActSpeed >= prMtr->trMtrType.u16Threshold2))
			{
				prMtr->trMtrAlgo.u8TblSpeed = 0;
			}
			
			if((prMtr->trMtrAlgo.s16ActSpeed >= prMtr->trMtrType.u16Threshold3) & (prMtr->trMtrAlgo.s16ActSpeed <= prMtr->trMtrType.u16Threshold4))
			{
				prMtr->trMtrAlgo.u8TblSpeed = 1;
			}
		}
		
		if(prMtr->trMtrAlgo.s16ActSpeed < 0)
		{
			prMtr->trMtrAlgo.eMtrCmdDir = SM_DIRECTION_CCW;
			
			if(((-prMtr->trMtrAlgo.s16ActSpeed) <= prMtr->trMtrType.u16Threshold1) | ((-prMtr->trMtrAlgo.s16ActSpeed) >= prMtr->trMtrType.u16Threshold2))
			{
				prMtr->trMtrAlgo.u8TblSpeed = 0;
			}
			
			if(((-prMtr->trMtrAlgo.s16ActSpeed) >= prMtr->trMtrType.u16Threshold3) & ((-prMtr->trMtrAlgo.s16ActSpeed) <= prMtr->trMtrType.u16Threshold4))
			{
				prMtr->trMtrAlgo.u8TblSpeed = 1;
			}
		}
		
		if(prMtr->trMtrAlgo.s16ActSpeed == 0)
		{
			prMtr->trMtrAlgo.u8TblSpeed = 0;
		}
		
		prMtr->trMtrAlgo.u32ActVelTimeout = prMtr->trGeneral.eUpdateTime;
	}
	
    return (prMtr->trMtrAlgo.u32ActVelTimeout);
}

/*******************************************************************************
* @function_name SM_HLL_Rtz
*//*
* @brief 		The function initializes Full stepping RTZ movement
* @param[in] 	prMtr - pointer to structure containing all the motor parameters
* @details 		The function downcounts the damping interval with every RTZ step.
*				When the damping interval is > 0, result status of the integration is 
*				marked as "invalid" and the integration result is ignored. When the 
*				damping interval is 0, result status of the integration is marked as 
*				"Valid" and the integration status is compared with the BEMF trigger.
* @note			The damping interval represents number of RTZ steps that are ignored 
*				at the beginning of every RTZ move. This interval should prevent the 
*				misdetection of zero during the run-up.  
*/
static void SM_HLL_RtzMove(SM_MotorParams_t *prMtr)
{
	/* Call function to execute next RTZ step */
	SM_LLL_Rtz(prMtr);	
}

/*******************************************************************************
* @function_name SM_HLL_MoveScheduler
*//*
* @brief 		The function schedules the motor movement
* @param[in] 	prMtr -	pointer to structure containing all the motor parameters
* @retval 		u32NextStepTimeout	- next ustep timeout
* @details 		When the motor is running and not frozen, function executes one 
*				algorithm iteration to reach required motor position. Otherwise 
*				the function just returns next step timeout according to the 
*				actual motor state. 
*/

static uint32_t SM_HLL_MoveScheduler(SM_MotorParams_t *prMtr)
{
    uint32_t u32NextStepTimeout;

	/*SM_Table_t *aa;
	aa = SMIA_ISM_STDPWMTABLE[0][0];*/

	/* If the motor is enabled */ 
	if(prMtr->trGeneral.eMotorEnable == SM_MTR_ENABLED)
	{
		/* Motor is running */
	    if(prMtr->trGeneral.eMotorState == SM_MOTOR_STATE_RUN)
	    {         
	    	/* Motor is not frozen, normal motor movement */
	        if(prMtr->trGeneral.eFreeze == SM_FREEZE_STATE_UNFREEZE)
	        {
	   			/* Execute motor algorithm */ 
	   			/* Next step timeout calculated by algorithm according to required motor velocity */
	          	u32NextStepTimeout = SM_HLL_MoveAlgorithm(prMtr);  
	          	      
	    		/* Update PWM values to move motor into the new position */
	    		SM_LLL_DrivePwms(prMtr);
	        }
	        
	        /* Motor is frozen */
	        else
	        {
	    		/* Next step timeout equals to the freeze timeout (1ms) */        
	        	u32NextStepTimeout = SM_MTR_FREEZE_TIMEOUT;
	        }
	    } 
	      
	    /* Rtz is running */    
	    else if(prMtr->trGeneral.eMotorState == SM_MOTOR_STATE_RTZ_RUN)
	    {
	    	/* Next step timeout equals to the Rtz timeout (1ms) */ 
	       	u32NextStepTimeout = SM_MTR_RTZ_TIMEOUT;
	    }
	    
	    /* Rtz is done */
	    else if(prMtr->trGeneral.eMotorState == SM_MOTOR_STATE_RTZ_DONE)
	    {
	    	/* Next step timeout equals to the Rtz done timeout (1ms) */ 
			u32NextStepTimeout = SM_MTR_RTZ_DONE_TIMEOUT;         	               
	    }  
	    
	    /* Motor is stopped */
	    else 
	    {
			/* Next step timeout equals to the Motor stop timeout (1ms) */ 
	    	u32NextStepTimeout = SM_MTR_STOP_TIMEOUT;
	    }
	}
	
	/* If the motor is disabled */ 
	else
	{
		/* Next step timeout equals to the Motor disable timeout (1ms) */ 
		u32NextStepTimeout = SM_MTR_DISABLED_TIMEOUT;	
	}
	
    return(u32NextStepTimeout);		/* Return next step timeout */    
}

/*==================================================================================================
                                       GLOBAL FUNCTIONS
==================================================================================================*/

/*******************************************************************************
* @function_name SM_InitMotorChannel
*//*
* @brief 		The function initializes the corresponding stepper motor
* @param[in] 	eMtrId	- stepper motor going to be controlled 
* @param[in] 	prMtrParams - initial parameters of the motor
* @retval 		Error code corresponding with SM_Err_t
* @details 		The function checks validity of critical input parameters. Once all
*				checked parameters are ok, the function initializes the main 
*				structure of the corresponding stepper motor. When trMtrType and 
*				trGeneral parameters are initialized the function initializes HW 
*				modules (SIUL, SMC) and stepper motor timer. The function also 
*				calculates the first step timeout from the maximum motor acceleration. 
*/
SM_Err_t SM_MotorInit(SM_MotorId_t eMtrId, HLL_MotorParams_t *prMtrParams)
{    
	float fAcc;
	SM_Err_t eErr;
	
	/* Default error code */
	eErr = SM_OK;
	
	/* If eMtrId > than maximum number of motors the eMtrId is invalid */
	if((eMtrId >= SM_MAX_MOTORS))
	{
		eErr = SM_ERR_INVALID_MTR_ID;	/* Invalid motor ID error code */
	}
	
	/* If motor ID is ok */ 
	else 
	{	
		/* If no error */
		if(eErr == SM_OK)
		{
			/* assign elements of the corresponding motor structure by the default values and 
			the initial parameters */    
		    
		    SM_HLL_rMtr[eMtrId].trMtrType.s16MotorMaxVel = prMtrParams->u16MtrMaxVel / 4u;
			SM_HLL_rMtr[eMtrId].trMtrType.s16MotorMaxAcc = prMtrParams->u16MtrMaxAcc / 16u;
			SM_HLL_rMtr[eMtrId].trMtrType.s16MotorMaxDec = prMtrParams->u16MtrMaxDec / 16u;
			SM_HLL_rMtr[eMtrId].trMtrAlgo.u16DampingFactor = prMtrParams->u16DampingFactor;
			SM_HLL_rMtr[eMtrId].trGeneral.eUpdateTime = prMtrParams->eUpdateTime;
			SM_HLL_rMtr[eMtrId].trMtrType.u16Threshold1 = prMtrParams->u16Threshold1 / 4u;
			SM_HLL_rMtr[eMtrId].trMtrType.u16Threshold2 = prMtrParams->u16Threshold2 / 4u;
			SM_HLL_rMtr[eMtrId].trMtrType.u16Threshold3 = prMtrParams->u16Threshold3 / 4u;
			SM_HLL_rMtr[eMtrId].trMtrType.u16Threshold4 = prMtrParams->u16Threshold4 / 4u;
					    
		    SM_HLL_rMtr[eMtrId].trGeneral.eMotorId = eMtrId;
		    SM_HLL_rMtr[eMtrId].trGeneral.eFreeze = SM_FREEZE_STATE_UNFREEZE;
		    SM_HLL_rMtr[eMtrId].trGeneral.s32ReqPos = 0u;
		    SM_HLL_rMtr[eMtrId].trGeneral.eMotorNominalDir = prMtrParams->eNominalDirection;
		    SM_HLL_rMtr[eMtrId].trGeneral.eZeroDetected = SM_ZERO_NOT_DETECTED;
		    /* Algorithm variables */
		   	SM_HLL_rMtr[eMtrId].trMtrAlgo.u16Step = 0u;
			SM_HLL_rMtr[eMtrId].trMtrAlgo.u16Ramp = 0u;	
			SM_HLL_rMtr[eMtrId].trMtrAlgo.eMtrCmdDir = SM_DIRECTION_CW;
		   	SM_HLL_rMtr[eMtrId].trMtrAlgo.u32ActVelTimeout = 0u;
		    SM_HLL_rMtr[eMtrId].trMtrAlgo.u32MaxVelTimeout = 0u;
		    SM_HLL_rMtr[eMtrId].trMtrAlgo.s32Pos = 0u;
		    SM_HLL_rMtr[eMtrId].trMtrAlgo.s32CmdPos = 0u;
		    SM_HLL_rMtr[eMtrId].trMtrAlgo.eAlgoState = SM_ALGO_STATE_STOP;
		    
		    SM_HLL_rMtr[eMtrId].trMtrAlgo.s16ActPos = 0u;
		    SM_HLL_rMtr[eMtrId].trMtrAlgo.s16TargetPos = 0u;
		    SM_HLL_rMtr[eMtrId].trMtrAlgo.s16TempCalc1 = 0u;
		    SM_HLL_rMtr[eMtrId].trMtrAlgo.s16ActSpeed = 0u;
		    SM_HLL_rMtr[eMtrId].trMtrAlgo.s16PrevSpeed = 0u;
		    SM_HLL_rMtr[eMtrId].trMtrAlgo.s16ActAcc = 0u;
		    SM_HLL_rMtr[eMtrId].trMtrAlgo.s16OldPos = 0u;
			   			 
			/* Copy parameters into trRtz structure */
			SM_HLL_rMtr[eMtrId].trRtz.eRtzStartQuad = prMtrParams->eRtzStartQuad;
			SM_HLL_rMtr[eMtrId].trRtz.u16RtzIntegrationTime = prMtrParams->u16RtzIntegTimeUs;
		    SM_HLL_rMtr[eMtrId].trRtz.u16RtzBlankingTime = prMtrParams->u16RtzBlankTimeUs;
		    
		    
		    switch(prMtrParams->eRtzBackEmfThreshold)
		    {
		    	case SM_RTZ_THRESHOLD_MV350:
		    		SM_HLL_rMtr[eMtrId].trRtz.s16RtzBemfThreshold = 10u;     
		    	break;
		    	
		    	case SM_RTZ_THRESHOLD_MV450:
		    		SM_HLL_rMtr[eMtrId].trRtz.s16RtzBemfThreshold = 30u;     
		    	break;
		    	
		    	case SM_RTZ_THRESHOLD_MV550:
		    		SM_HLL_rMtr[eMtrId].trRtz.s16RtzBemfThreshold = 50u;     
		    	break;
		    	
		    	case SM_RTZ_THRESHOLD_MV650:
		    		SM_HLL_rMtr[eMtrId].trRtz.s16RtzBemfThreshold = 100u;     
		    	break;
		    	
		    	case SM_RTZ_THRESHOLD_MV750:
		    		SM_HLL_rMtr[eMtrId].trRtz.s16RtzBemfThreshold = 200u;     
		    	break;
		    	
		    	case SM_RTZ_THRESHOLD_MV850:
		    		SM_HLL_rMtr[eMtrId].trRtz.s16RtzBemfThreshold = 300u;     
		    	break;
		    	
		    	default:     
		    	break;
		    }
		    
		    SM_HLL_rMtr[eMtrId].trGeneral.eMotorState = SM_MOTOR_STATE_STOP;
		    SM_HLL_rMtr[eMtrId].trGeneral.eMotorEnable = SM_MTR_ENABLED;
		    		    
		    SM_HLL_rMtr[eMtrId].trGeneral.cpTableSet = pa2TableSet[prMtrParams->eMtrModel][prMtrParams->eNominalDirection];
		    
		    SM_HLL_rMtr[eMtrId].trGeneral.eAlgoType = prMtrParams->eAlgoType;
		    
		    /* Initialize HW modules (SIUL and SMC) */
		    SM_HLL_InitHwModules(eMtrId); 
		    /* Initialize motor timer */ 
		    SM_HLL_TimerInit(eMtrId); 
		}
	}
	
    return eErr;	/* Return error code */		
}

/*******************************************************************************
* @function_name SM_MotorDeInit
*//*
* @brief 		The function disables the corresponding stepper motor
* @param[in] 	eMtrId	- stepper motor going to be controlled 
* @retval 		Error code corresponding with SM_Err_t
* @details 		If the stepper motor is initialized, the function disables
*				the corresponding stepper motor  
*/
SM_Err_t SM_MotorDeInit(SM_MotorId_t eMtrId)
{
	SM_Err_t eErr;
	
	/* Default error code */
	eErr = SM_OK;
	
	/* If eMtrId > than maximum number of motors the eMtrId is invalid */
	if((eMtrId >= SM_MAX_MOTORS))
	{
		eErr = SM_ERR_INVALID_MTR_ID;	/* Invalid motor ID error code */
	}
	
	/* If motor ID is ok */ 
	else
	{
		/* If the stepper motor is not enabled, motor not enabled error */
		if(SM_HLL_rMtr[eMtrId].trGeneral.eMotorEnable != SM_MTR_ENABLED)
		{
			eErr = SM_ERR_MTR_NOT_ENABLED;	/* Stepper motor not enabled error */
		}
		
		/* If no error, disable stepper motor */ 
		if(eErr == SM_OK)
		{
			SM_HLL_rMtr[eMtrId].trGeneral.eMotorEnable = SM_MTR_DISABLED;	
		}
		
		/* add - disable HW modules */ 
	}
	
	return eErr;	/* Return error code */
}

/*******************************************************************************
* @function_name SM_GoToPosition
*//*
* @brief 		The function initializes the motor movement into a new position
* @param[in] 	eMtrId	- stepper motor going to be controlled
* @param[in] 	s32Pos	- required position [rad] - scaled (0 = 0pi [rad]; 
*				32768 = 1 pi [rad]; 65536 = 0(2) pi [rad]...)
* @param[in] 	u32Vel	- required velocity [rad/s] - scaled
* @retval 		Error code corresponding with SM_Err_t
* @details 		The function sets up the new motor movement. Calculates new 
*				required position and velocity to move the rotor. Function also
*				commands the algorithm to initialize the new motor movement.
*/

SM_Err_t SM_GoToPosition(SM_MotorId_t eMtrId, uint16_t u16Pos)
{
	float fPos,fVel;
	SM_Err_t eErr;
	
	/* Default error code */
	eErr = SM_OK;
	
	/* If the eMtrId > than maximum number of motors the eMtrId is invalid */
	if((eMtrId >= SM_MAX_MOTORS))
	{
		eErr = SM_ERR_INVALID_MTR_ID;	/* Invalid motor ID error code */
	}
	
	/* If the motor ID is ok */ 
	else
	{	
	
		/* If the stepper motor not enabled, motor not enabled error */
		if(SM_HLL_rMtr[eMtrId].trGeneral.eMotorEnable != SM_MTR_ENABLED)
		{
			eErr = SM_ERR_MTR_NOT_ENABLED;	/* Stepper motor not enabled error */
		}	
		
		/* If no error, disable stepper motor */		
		if(eErr == SM_OK)
		{			
			SM_HLL_rMtr[eMtrId].trGeneral.s32ReqPos = u16Pos;	
					
			/* Commant to the algorithm to initialize the new motor movement */			
			//SM_HLL_rMtr[eMtrId].trGeneral.eCmdToAlgo = SM_CMD_TO_ALGO_UPDATE;

			/* Update motor status to MOTOR RUN */				
			SM_HLL_rMtr[eMtrId].trGeneral.eMotorState = SM_MOTOR_STATE_RUN;
		}	
	}
	
	return eErr;	/* Return error code */
}

/*******************************************************************************
* @function_name SM_GetActualVariable
*//*
* @brief 		The function ... tbd 
* @param[in] 	eMtrId	- stepper motor going to be inspected
* @param[in] 	
* @retval 		Error code corresponding with SM_Err_t
* @details 		
*/
SM_Err_t SM_GetActualVariable(SM_MotorId_t eMtrId, SM_ActualVariable_t eActVar, int16_t *ps16Var)
{
	SM_Err_t eErr;
	
	/* Default error code */
	eErr = SM_OK;
	
	/* If the eMtrId > than maximum number of motors the eMtrId is invalid */
	if((eMtrId >= SM_MAX_MOTORS))
	{
		eErr = SM_ERR_INVALID_MTR_ID;	/* Invalid motor ID error code */
	}
	
	/* If the motor ID is ok */ 
	else
	{	
		/* If the stepper motor not enabled, motor not enabled error */
		if(SM_HLL_rMtr[eMtrId].trGeneral.eMotorEnable != SM_MTR_ENABLED)
		{
			eErr = SM_ERR_MTR_NOT_ENABLED;	/* Stepper motor not enabled error */
		}
		
		/* If no error */
		if(eErr == SM_OK)
		{
			switch(eActVar)
			{
				case SM_ACTUAL_POSITION:
					*ps16Var = SM_HLL_rMtr[eMtrId].trMtrAlgo.s16ActPos;	
					/* unify algo variables or decide which algo position is returned */
				break;
				
				case SM_ACTUAL_SPEED:
					*ps16Var = SM_HLL_rMtr[eMtrId].trMtrAlgo.s16ActSpeed;
				break;
				
				case SM_ACTUAL_ACCELERATION:
					*ps16Var = SM_HLL_rMtr[eMtrId].trMtrAlgo.s16ActAcc;				
				break;
								
				case SM_ACTUAL_DAMPING_FACTOR:
					*ps16Var = SM_HLL_rMtr[eMtrId].trMtrAlgo.u16DampingFactor;				
				break;				
				
				case SM_ACTUAL_TARGET_POSITION:
					*ps16Var = SM_HLL_rMtr[eMtrId].trMtrAlgo.s16TargetPos;				
				break;
								
				case SM_ACTUAL_ACCELERATION_LIMIT:
					*ps16Var = SM_HLL_rMtr[eMtrId].trMtrType.s16MotorMaxAcc;				
				break;
								
				case SM_ACTUAL_DECELERATION_LIMIT:
					*ps16Var = SM_HLL_rMtr[eMtrId].trMtrType.s16MotorMaxDec;				
				break;
								
				case SM_ACTUAL_MAXIMUM_SPEED:
					*ps16Var = SM_HLL_rMtr[eMtrId].trMtrType.s16MotorMaxVel;				
				break;
								
				case SM_ACTUAL_THRESHOLD_1:
					*ps16Var = SM_HLL_rMtr[eMtrId].trMtrType.u16Threshold1;				
				break;				
				
				case SM_ACTUAL_THRESHOLD_2:
					*ps16Var = SM_HLL_rMtr[eMtrId].trMtrType.u16Threshold2;				
				break;				
				
				case SM_ACTUAL_THRESHOLD_3:
					*ps16Var = SM_HLL_rMtr[eMtrId].trMtrType.u16Threshold3;				
				break;			
				
				case SM_ACTUAL_THRESHOLD_4:
					*ps16Var = SM_HLL_rMtr[eMtrId].trMtrType.u16Threshold4;				
				break;
				
				default:
				break;
			}
		}
	}

	return eErr;	/* Return error code */
}

/*******************************************************************************
* @function_name SM_SetActualVariable
*//*
* @brief 		The function ... tbd 
* @param[in] 	eMtrId	- stepper motor going to be inspected
* @param[in] 	
* @retval 		Error code corresponding with SM_Err_t
* @details 		
*/
SM_Err_t SM_SetActualVariable(SM_MotorId_t eMtrId, SM_ActualVariable_t eActVar, int16_t s16Var)
{
	SM_Err_t eErr;
	
	/* Default error code */
	eErr = SM_OK;
	
	/* If the eMtrId > than maximum number of motors the eMtrId is invalid */
	if((eMtrId >= SM_MAX_MOTORS))
	{
		eErr = SM_ERR_INVALID_MTR_ID;	/* Invalid motor ID error code */
	}
	
	/* If the motor ID is ok */ 
	else
	{	
		/* If the motor not frozen, motor not frozen error */
		if(SM_HLL_rMtr[eMtrId].trGeneral.eFreeze == SM_FREEZE_STATE_UNFREEZE)
		{
			eErr = SM_ERR_MTR_NOT_FROZEN;	/* Motor not frozen error */
		}
		
		/* If the stepper motor not enabled, motor not enabled error */
		if(SM_HLL_rMtr[eMtrId].trGeneral.eMotorEnable != SM_MTR_ENABLED)
		{
			eErr = SM_ERR_MTR_NOT_ENABLED;	/* Stepper motor not enabled error */
		}
		
		/* If no error */
		if(eErr == SM_OK)
		{
			switch(eActVar)
			{
				case SM_ACTUAL_POSITION:
					SM_HLL_rMtr[eMtrId].trMtrAlgo.s16ActPos = s16Var;	
					/* unify algo variables or decide which algo position is returned */
				break;
				
				case SM_ACTUAL_SPEED:
					SM_HLL_rMtr[eMtrId].trMtrAlgo.s16ActSpeed = s16Var;
				break;
				
				case SM_ACTUAL_ACCELERATION:
					SM_HLL_rMtr[eMtrId].trMtrAlgo.s16ActAcc = s16Var;				
				break;
								
				case SM_ACTUAL_DAMPING_FACTOR:
					SM_HLL_rMtr[eMtrId].trMtrAlgo.u16DampingFactor = s16Var;				
				break;				
				
				case SM_ACTUAL_TARGET_POSITION:
					SM_HLL_rMtr[eMtrId].trMtrAlgo.s16TargetPos = s16Var;				
				break;
								
				case SM_ACTUAL_ACCELERATION_LIMIT:
					SM_HLL_rMtr[eMtrId].trMtrType.s16MotorMaxAcc = s16Var;				
				break;
								
				case SM_ACTUAL_DECELERATION_LIMIT:
					SM_HLL_rMtr[eMtrId].trMtrType.s16MotorMaxDec = s16Var;				
				break;
								
				case SM_ACTUAL_MAXIMUM_SPEED:
					SM_HLL_rMtr[eMtrId].trMtrType.s16MotorMaxVel = s16Var;				
				break;
								
				case SM_ACTUAL_THRESHOLD_1:
					SM_HLL_rMtr[eMtrId].trMtrType.u16Threshold1 = s16Var;				
				break;				
				
				case SM_ACTUAL_THRESHOLD_2:
					SM_HLL_rMtr[eMtrId].trMtrType.u16Threshold2 = s16Var;				
				break;				
				
				case SM_ACTUAL_THRESHOLD_3:
					SM_HLL_rMtr[eMtrId].trMtrType.u16Threshold3 = s16Var;				
				break;			
				
				case SM_ACTUAL_THRESHOLD_4:
					SM_HLL_rMtr[eMtrId].trMtrType.u16Threshold4 = s16Var;				
				break;
				
				default:
				break;
			}
		}
	}

	return eErr;	/* Return error code */
}

/*******************************************************************************
* @function_name SM_SetCoilOutputs
*//*
* @brief 		The function ... tbd 
* @param[in] 	eMtrId	- stepper motor going to be inspected
* @param[in] 	
* @retval 		Error code corresponding with SM_Err_t
* @details 		
*/
SM_Err_t SM_SetCoilOutputs(SM_MotorId_t eMtrId, uint16_t u16CosPwm, uint16_t u16SinPwm, SM_RtzQuadrant_t eQuad)
{
	SM_Err_t eErr;
	
	/* Default error code */
	eErr = SM_OK;
	
	/* If the eMtrId > than maximum number of motors the eMtrId is invalid */
	if((eMtrId >= SM_MAX_MOTORS))
	{
		eErr = SM_ERR_INVALID_MTR_ID;	/* Invalid motor ID error code */
	}
	
	/* If the motor ID is ok */ 
	else
	{	
		/* If the motor not frozen, motor not frozen error */
		if(SM_HLL_rMtr[eMtrId].trGeneral.eFreeze == SM_FREEZE_STATE_UNFREEZE)
		{
			eErr = SM_ERR_MTR_NOT_FROZEN;	/* Motor not frozen error */
		}
		
		/* If the stepper motor not enabled, motor not enabled error */
		if(SM_HLL_rMtr[eMtrId].trGeneral.eMotorEnable != SM_MTR_ENABLED)
		{
			eErr = SM_ERR_MTR_NOT_ENABLED;	/* Stepper motor not enabled error */
		}
		
		/* If no error */
		if(eErr == SM_OK)
		{
			/*tbd*/
		}
	}

	return eErr;	/* Return error code */
}

/*******************************************************************************
* @function_name SM_ReadMotorStatus
*//*
* @brief 		The function provides actual motor status.
* @param[in] 	eMtrId	- stepper motor going to be inspected  
* @param[in] 	peMtrStatus	- pointer to an SM_MotorState_t where the motor 
*				status will be saved
* @retval 		Error code corresponding with SM_Err_t
* @details 		The function copies the motor status onto provided address. 
*				The status can be:
*				SM_MOTOR_STOP		- motor doesn't move and the position is random 
*				SM_MOTOR_RUN		- motor currently operates normal movement
*				SM_MOTOR_RTZ_RUN	- motor currently operates RTZ movement
*				SM_MOTOR_RTZ_DONE	- motor just found zero position (stall)
*/

SM_Err_t SM_ReadMtrStatus(SM_MotorId_t eMtrId,SM_MotorState_t *peMtrStatus)
{
	SM_Err_t eErr;
	
	/* Default error code */
	eErr = SM_OK;
	
	/* If the eMtrId > than maximum number of motors the eMtrId is invalid */
	if((eMtrId >= SM_MAX_MOTORS))
	{
		eErr = SM_ERR_INVALID_MTR_ID;	/* Invalid motor ID error code */
	}
	
	/* If the motor ID is ok */ 
	else
	{
		/* If the stepper motor not enabled, motor not enabled error */	
		if(SM_HLL_rMtr[eMtrId].trGeneral.eMotorEnable != SM_MTR_ENABLED)
		{
			eErr = SM_ERR_MTR_NOT_ENABLED;	/* Stepper motor not enabled error */
		}
		
		/* If no error */
		if(eErr == SM_OK)
		{
			/* Copy motor status from corresponding motor structure onto the provided address */
			*peMtrStatus = SM_HLL_rMtr[eMtrId].trGeneral.eMotorState;
		}
	}

	return eErr;	/* Return error code */
}

/*******************************************************************************
* @function_name SM_Disable  ( old freeze function)
*//*
* @brief 		The function freezes motor 
* @param[in] 	eMtrId	- stepper motor going to be frozen  
* @param[in] 	eFreezeMode	- freeze mode 
* @details		Function Updates "eFreezeMode" in the "trMtrAlgo" structure according
*				to the required freeze mode and commands algorithm to ramp donw and 
*				freeze the movement.   		
*/

SM_Err_t SM_Disable(SM_MotorId_t eMtrId)
{
	SM_Err_t eErr;
	
	/* Default error code */
	eErr = SM_OK;
	
	/* If the eMtrId > than maximum number of motors the eMtrId is invalid */
	if((eMtrId >= SM_MAX_MOTORS))
	{
		eErr = SM_ERR_INVALID_MTR_ID;	/* invalid motor ID error code */
	}
	
	/* If the motor ID is ok */ 
	else
	{
		/* If the motor is already frozen, return motor already frozen error */
		if(SM_HLL_rMtr[eMtrId].trGeneral.eFreeze == SM_FREEZE_STATE_FREEZE)
		{
			eErr = SM_ERR_MTR_ALREADY_FROZEN;	/* Motor already frozen error */
		}
							
		/* If the stepper motor not enabled, motor not enabled error */
		if(SM_HLL_rMtr[eMtrId].trGeneral.eMotorEnable != SM_MTR_ENABLED)
		{
			eErr = SM_ERR_MTR_NOT_ENABLED;	/* Stepper motor not enabled error */
		}
	
		/* If no error */
		if(eErr == SM_OK)
		{
			SM_HLL_rMtr[eMtrId].trGeneral.eFreeze = SM_FREEZE_STATE_FREEZE;		
		}
	}
	
	return eErr;	/* Return error code */
}

/*******************************************************************************
* @function_name SM_ReEnable
*//*
* @brief 		The function unfreezes the motor (motor will continue in the 
*				previous movement)
* @param[in] 	eMtrId	- stepper motor going to be unfrozen  
* @details 		Function commands algorithm to update, sets motor coils to the 
*				"full power" and unfreezes. 
*/

SM_Err_t SM_ReEnable(SM_MotorId_t eMtrId)
{
	SM_Err_t eErr;
	
	/* Default error code */
	eErr = SM_OK;
	
	/* If the eMtrId > than maximum number of motors the eMtrId is invalid */
	if((eMtrId >= SM_MAX_MOTORS))
	{
		eErr = SM_ERR_INVALID_MTR_ID;	/* Invalid motor ID error code */
	}
	
	/* If the motor ID is ok */ 
	else
	{
		/* If the motor not frozen, motor not frozen error */
		if(SM_HLL_rMtr[eMtrId].trGeneral.eFreeze == SM_FREEZE_STATE_UNFREEZE)
		{
			eErr = SM_ERR_MTR_NOT_FROZEN;	/* Motor not frozen error */
		}
		
		/* If the stepper motor not enabled, motor not enabled error */
		if(SM_HLL_rMtr[eMtrId].trGeneral.eMotorEnable != SM_MTR_ENABLED)
		{
			eErr = SM_ERR_MTR_NOT_ENABLED;	/* Stepper motor not enabled error */
		}
		
		/* If no error */		
		if(eErr == SM_OK)
		{
			SM_HLL_rMtr[eMtrId].trGeneral.eFreeze = SM_FREEZE_STATE_UNFREEZE;
		}
	}
	
	return eErr;	/* Return error code */
}

/*******************************************************************************
* @function_name SM_Rtz
*//*
* @brief 		The function initializes RTZ movement to find the stopper and to 
*				determine zero position (recalibration of the system of 
*				coordinates orientation)		
* @param[in] 	eMtrId	- stepper motor going to be controlled  
* @param[in] 
* @retval 		Error code corresponding with SM_Err_t 
* @details 		The function initializes RTZ parameters according to the required
*				and user defined RTZ parameters. Calculates integration time and 
*				sets motor into the RTZ RUN state.
*/

SM_Err_t SM_Rtz(SM_MotorId_t eMtrId)
{
	SM_Err_t eErr;
	
	/* Default error code */
	eErr = SM_OK;
	
	/* If the eMtrId > than maximum number of motors the eMtrId is invalid */
	if((eMtrId >= SM_MAX_MOTORS))
	{
		eErr = SM_ERR_INVALID_MTR_ID;	/* invalid motor ID error code */
	}
	
	/* If the motor ID is ok */ 
	else
	{			
		/* If the stepper motor not enabled, motor not enabled error */
		if(SM_HLL_rMtr[eMtrId].trGeneral.eMotorEnable != SM_MTR_ENABLED)
		{
			eErr = SM_ERR_MTR_NOT_ENABLED;	/* Stepper motor not enabled error */
		}
		
		/* If no error */
		if(eErr == SM_OK)
		{
			/* Mark integration result as invalid */	
		    SM_HLL_rMtr[eMtrId].trRtz.eIntResultStat = SM_HLL_INT_RESULT_INVALID; 	
		    
		    /* Update Motor state to RTZ RUN */ 
		    SM_HLL_rMtr[eMtrId].trGeneral.eMotorState = SM_MOTOR_STATE_RTZ_RUN; 
		    
		    SM_HLL_rMtr[eMtrId].trGeneral.eZeroDetected = SM_ZERO_NOT_DETECTED;
		    
		    /* Initialize SSD module */               		
			SM_HLL_SsdInit(eMtrId);
			
			/* RTZ first step */
			SM_HLL_RtzMove(&SM_HLL_rMtr[eMtrId]);
		}
	}
	
	return eErr;	/* Return error code */
}

/*******************************************************************************
* @function_name SM_GetActualStopQuadrant
*//*
* @brief 				
* @param[in] 	eMtrId	- stepper motor going to be controlled  
* @param[in] 	
* @retval 		Error code corresponding with SM_Err_t 
* @details 		
*/

SM_Err_t SM_GetActualStopQuadrant(SM_MotorId_t eMtrId, SM_RtzQuadrant_t *eQuadrant)
{
	SM_Err_t eErr;
	
	/* Default error code */
	eErr = SM_OK;
	
	/* If the eMtrId > than maximum number of motors the eMtrId is invalid */
	if((eMtrId >= SM_MAX_MOTORS))
	{
		eErr = SM_ERR_INVALID_MTR_ID;	/* invalid motor ID error code */
	}
	
	/* If the motor ID is ok */ 
	else
	{		
		/* If the stepper motor not enabled, motor not enabled error */
		if(SM_HLL_rMtr[eMtrId].trGeneral.eMotorEnable != SM_MTR_ENABLED)
		{
			eErr = SM_ERR_MTR_NOT_ENABLED;	/* Stepper motor not enabled error */
		}
		
		if(SM_HLL_rMtr[eMtrId].trGeneral.eZeroDetected == SM_ZERO_NOT_DETECTED)
		{
			eErr = SM_ERR_ZERO_NOT_DETECTED;
		}
		
		/* If no error */
		if(eErr == SM_OK)
		{
			*eQuadrant = SM_HLL_rMtr[eMtrId].trRtz.eRtzStopQuad;
		}
	}
}


/*==================================================================================================
                                       INTERRUPT HANDLERS
==================================================================================================*/

void STM0Ch0ISR(void)
{
  	STM_0.CHANNEL[0].CIR.R = 0x1;			/* Channel 0 Interrupt cleared */

  	/* Schedule new timer interrupt, the timeout is provided by SM_HLL_MoveScheduler */ 
  	LLL_TIMER0_VALUE(SM_MOTOR_ID_1) = LLL_TIMER0_VALUE(SM_MOTOR_ID_1) 
  	+ SM_HLL_MoveScheduler(&SM_HLL_rMtr[SM_MOTOR_ID_1]);		
}

void STM0Ch1ISR(void)
{ 
  	STM_0.CHANNEL[1].CIR.R = 0x1;			/* Channel 1 Interrupt cleared */
  	
  	/* Schedule new timer interrupt, the timeout is provided by SM_HLL_MoveScheduler */ 
  	LLL_TIMER0_VALUE(SM_MOTOR_ID_2) = LLL_TIMER0_VALUE(SM_MOTOR_ID_2) 
  	+ SM_HLL_MoveScheduler(&SM_HLL_rMtr[SM_MOTOR_ID_2]);			
}

void STM0Ch2ISR(void)
{ 
  	STM_0.CHANNEL[2].CIR.R = 0x1;			/* Channel 2 Interrupt cleared */
  	
  	/* Schedule new timer interrupt, the timeout is provided by SM_HLL_MoveScheduler */ 
  	LLL_TIMER0_VALUE(SM_MOTOR_ID_3) = LLL_TIMER0_VALUE(SM_MOTOR_ID_3) 
  	+ SM_HLL_MoveScheduler(&SM_HLL_rMtr[SM_MOTOR_ID_3]);
}

void STM0Ch3ISR(void)
{ 
  	STM_0.CHANNEL[3].CIR.R = 0x1;			/* Channel 3 Interrupt cleared */
  	
  	/* Schedule new timer interrupt, the timeout is provided by SM_HLL_MoveScheduler */ 
  	LLL_TIMER0_VALUE(SM_MOTOR_ID_4) = LLL_TIMER0_VALUE(SM_MOTOR_ID_4) 
  	+ SM_HLL_MoveScheduler(&SM_HLL_rMtr[SM_MOTOR_ID_4]);
}

void STM1Ch0ISR(void)
{ 
  	STM_1.CHANNEL[0].CIR.R = 0x1;			/* Channel 4 Interrupt cleared */

	/* Schedule new timer interrupt, the timeout is provided by SM_HLL_MoveScheduler */ 
  	LLL_TIMER1_VALUE(SM_MOTOR_ID_5) = LLL_TIMER1_VALUE(SM_MOTOR_ID_5) 
  	+ SM_HLL_MoveScheduler(&SM_HLL_rMtr[SM_MOTOR_ID_5]);
}

void STM1Ch1ISR(void)
{ 
  	STM_1.CHANNEL[1].CIR.R = 0x1;			/* Channel 5 Interrupt cleared */

	/* Schedule new timer interrupt, the timeout is provided by SM_HLL_MoveScheduler */ 
  	LLL_TIMER1_VALUE(SM_MOTOR_ID_6) = LLL_TIMER1_VALUE(SM_MOTOR_ID_6) 
  	+ SM_HLL_MoveScheduler(&SM_HLL_rMtr[SM_MOTOR_ID_6]);		
}

void STM0ISR(void)
{
	uint32_t test;
	if (STM_0.CHANNEL[0].CIR.R == 0x1)	STM0Ch0ISR();
	if (STM_0.CHANNEL[1].CIR.R == 0x1)	STM0Ch1ISR();
	if (STM_0.CHANNEL[2].CIR.R == 0x1)	STM0Ch2ISR();
	if (STM_0.CHANNEL[3].CIR.R == 0x1)	STM0Ch3ISR();
}

void STM1ISR(void)
{
	if (STM_1.CHANNEL[0].CIR.R == 0x1)	STM1Ch0ISR();
	if (STM_1.CHANNEL[1].CIR.R == 0x1)	STM1Ch1ISR();
}


/* SSD 0 Integration time expired interrupt */
void _isr_SSD0_Interrupt(void)
{
	/* SSD interrupt service */
	SM_LLL_SsdInterrupt(&SM_HLL_rMtr[SM_MOTOR_ID_1]);
	
	/* Execute RTZ step */ 
	SM_HLL_RtzMove(&SM_HLL_rMtr[SM_MOTOR_ID_1]);
}

/* SSD 1 Integration time expired interrupt */
void _isr_SSD1_Interrupt(void)
{
	/* SSD interrupt service */
	SM_LLL_SsdInterrupt(&SM_HLL_rMtr[SM_MOTOR_ID_2]);
	
	/* Execute RTZ step */ 
	SM_HLL_RtzMove(&SM_HLL_rMtr[SM_MOTOR_ID_2]);
}

/* SSD 2 Integration time expired interrupt */
void _isr_SSD2_Interrupt(void)
{
	/* SSD interrupt service */
	SM_LLL_SsdInterrupt(&SM_HLL_rMtr[SM_MOTOR_ID_3]);
	
	/* Execute RTZ step */ 
	SM_HLL_RtzMove(&SM_HLL_rMtr[SM_MOTOR_ID_3]);
}

/* SSD 3 Integration time expired interrupt */
void _isr_SSD3_Interrupt(void)
{
	/* SSD interrupt service */
	SM_LLL_SsdInterrupt(&SM_HLL_rMtr[SM_MOTOR_ID_4]);
	
	/* Execute RTZ step */ 
	SM_HLL_RtzMove(&SM_HLL_rMtr[SM_MOTOR_ID_4]);
}

/* SSD 4 Integration time expired interrupt */
void _isr_SSD4_Interrupt(void)
{
	/* SSD interrupt service */
	SM_LLL_SsdInterrupt(&SM_HLL_rMtr[SM_MOTOR_ID_5]);
	
	/* Execute RTZ step */ 
	SM_HLL_RtzMove(&SM_HLL_rMtr[SM_MOTOR_ID_5]);
}

/* SSD 5 Integration time expired interrupt */
void _isr_SSD5_Interrupt(void)
{
	/* SSD interrupt service */
	SM_LLL_SsdInterrupt(&SM_HLL_rMtr[SM_MOTOR_ID_6]);
	
	/* Execute RTZ step */ 
	SM_HLL_RtzMove(&SM_HLL_rMtr[SM_MOTOR_ID_6]);
}

/*================================================================================================*/
#ifdef __cplusplus
}
#endif
